// Copyright Nuno Afonso

#include "ArkSourceControlProvider.h"

#include "ArkSourceControl.h"
#include "FArkSourceControlState.h"
#include "FArkSourceControlChangelistState.h"
#include "SArkSourceControlSettings.h"
#include "SourceControlHelpers.h"
#include "SourceControlOperations.h"
#include "Interfaces/IPluginManager.h"
#include "Widgets/Input/SButton.h"
#include "Framework/Notifications/NotificationManager.h"
#include "Widgets/Notifications/SNotificationList.h"
#include "ContentBrowserModule.h"
#include "Modules/ModuleManager.h"
#include "IContentBrowserSingleton.h"
#include "GenericPlatform/GenericPlatformFile.h"

#define LOCTEXT_NAMESPACE "ArkSourceControl"

#define LOG FMessageLog("SourceControl")

#define ARK_PLUGIN_DEBUG 0

void* FArkSourceControlProvider::plugin_alloc(int64_t size) {
	return FMemory::Malloc(size);
}
void FArkSourceControlProvider::plugin_free(void* memory) {
	FMemory::Free(memory);
}
void* FArkSourceControlProvider::plugin_realloc(void* old_memory, int64_t size, int64_t old_size) {
	return FMemory::Realloc(old_memory, size);
}
void FArkSourceControlProvider::plugin_logger(Ark_Plugin_String message, uint8_t log_level) {
	FString message_str = ARK_PLUGIN_STRING_TO_FSTRING(message);
	FText message_text = FText::FromString(message_str);
	switch (log_level) {
		case Log_Level_ERROR:
			LOG.Error(message_text);
			break;
		case Log_Level_WARN:
			LOG.Warning(message_text);
			break;
		default: // Log_Level_INFO and Log_Level_DEBUG
			LOG.Info(message_text);
			break;
	}
}

void FArkSourceControlProvider::plugin_response(Ark_Plugin_Response *response) {
	FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
	uint64_t command_id = 0;
	int32 request_index = -1;
	for (int32 i = 0; i != provider.requests.Num(); ++i) {
		if (provider.requests[i].request_id == response->request_id) {
			request_index = i;
			command_id = provider.requests[i].command_id;
			break;
		}
	}
	FArkSourceControlCommand* command = nullptr;
	for (int32 i = 0; i != provider.commands.Num(); ++i) {
		if (provider.commands[i]->id == command_id) {
			command = provider.commands[i];
			break;
		}
	}
	if (command) {
		command->handle_response(response);
	}
	else {
		LOG.Warning(FText::FromString(FString::Printf(TEXT("Received response for request id %ul but couldn't find the command it belongs to (%ul)"), response->request_id, command_id)));
	}

	if (request_index >= 0) {
		if (!ensure(request_index < provider.requests.Num())) {
			return;
		}
		provider.requests.RemoveAtSwap(request_index);
	}
}

void FArkSourceControlProvider::plugin_notification(Ark_Plugin_Notification *notification) {
	FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
	switch (notification->type) {
		case Notification_Type_WORKSPACE_INFO:
			// Nothing to do atm, workspace_info gets saved in the plugin, just get it via ark_plugin_c_workspace_info
		{
			auto& settings = FArkSourceControlModule::Get().settings;
	
			Ark_Plugin_Workspace_Info workspace_info = ark_plugin_c_workspace_info(provider.plugin);

			TCHAR Buffer[256];

			FStringBuilderBase sb = FStringBuilderBase(&Buffer[0], 256);
			sb.Appendf(TEXT("Workspace: %s"), *provider.get_workspace_full_path());
			sb.Appendf(TEXT("\nHost: %s"), *settings.host);
			sb.Appendf(TEXT("\nEmail: %s"), *settings.email);
			sb.Appendf(TEXT("\nBranch: %s"), *ARK_PLUGIN_STRING_TO_FSTRING(workspace_info.branch_str));
			sb.Appendf(TEXT("\nHead CL: %d"), workspace_info.head_cl_id);
			sb.Appendf(TEXT("\nCurrent CL: %d"), workspace_info.current_cl_id);

			if (workspace_info.committed_cl_ids.count > 0) {
				sb.Appendf(TEXT(" (%d"), *workspace_info.committed_cl_ids.data);
				for (int32 i = 1; i < workspace_info.committed_cl_ids.count; ++i) {
					sb.Appendf(TEXT(", %d"), *(workspace_info.committed_cl_ids.data + i));
				}
				sb.Append(TEXT(")"));
			}

			provider.status_text = FText::FromString(sb.ToString());

			// Request an update of all the states on our current directory so we get an immediate feedback
			provider.force_update_current_directory_relative_paths();
		}
			break;

		case Notification_Type_FILES_CHANGED:

			for (int32 i = 0; i < notification->files_changed.count; ++i) {
				Ark_Plugin_File_Changed* file_changed = notification->files_changed.data + i;
				auto* file_state = provider.get_file_state(ARK_PLUGIN_STRING_TO_FSTRING(file_changed->relative_path));
				file_state->update(file_changed->file);
			}

			break;
		case Notification_Type_LOCK_REQUEST:
		{
			uint64_t notification_id = notification->id;
			auto& notification_manager = FSlateNotificationManager::Get();
			FString relative_path = ARK_PLUGIN_STRING_TO_FSTRING(notification->lock_request.relative_path);
			FString requester = ARK_PLUGIN_STRING_TO_FSTRING(notification->lock_request.requester);
			FNotificationInfo info(FText::FromString(FString::Printf(TEXT("%s requested lock"), *requester)));
			info.FadeInDuration = 0.25f;
			info.FadeOutDuration = 0.25f;
			info.ExpireDuration = 30;
			info.SubText = FText::FromString(relative_path);
			info.ButtonDetails.Add(FNotificationButtonInfo(FText::FromString("Give Lock"), FText::FromString("Give the lock ownership to the requesting user"), FSimpleDelegate::CreateLambda([notification_id, relative_path, requester]() {
				FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
				ark_plugin_c_request_lock_give(provider.plugin, ark_plugin_c_string_talloc(provider.plugin, TCHAR_TO_UTF8(*relative_path)), ark_plugin_c_string_talloc(provider.plugin, TCHAR_TO_UTF8(*requester)), true);
				provider.set_notification_state(notification_id, SNotificationItem::CS_Success);
			})));
			info.ButtonDetails.Add(FNotificationButtonInfo(FText::FromString("Reject"), FText::FromString("Keep the lock ownership"), FSimpleDelegate::CreateLambda([notification_id, relative_path, requester]() {
				FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
				ark_plugin_c_request_lock_give(provider.plugin, ark_plugin_c_string_talloc(provider.plugin, TCHAR_TO_UTF8(*relative_path)), ark_plugin_c_string_talloc(provider.plugin, TCHAR_TO_UTF8(*requester)), false);
				provider.set_notification_state(notification_id, SNotificationItem::CS_Fail);
			})));
			auto notification_item = notification_manager.AddNotification(info);
			notification_item->SetCompletionState(SNotificationItem::CS_Pending);
			provider.notifications.Add( { notification_id, notification_item } );
		}
			break;
		case Notification_Type_LOCK_RESPONSE:
		{
			uint64_t notification_id = notification->id;
			auto& notification_manager = FSlateNotificationManager::Get();
			FString relative_path = ARK_PLUGIN_STRING_TO_FSTRING(notification->lock_response.relative_path);
			FString responder = ARK_PLUGIN_STRING_TO_FSTRING(notification->lock_response.responder);
			FString notification_header;
			if (responder.IsEmpty()) {
				if (notification->lock_response.accepted) {
					notification_header = TEXT("Received lock ownership");
				}
			}
			else {
				notification_header = FString::Printf(TEXT("%s %s"), *responder, notification->lock_response.accepted ? TEXT("gave you lock ownership") : TEXT("cannot give you lock ownership"));
			}
			if (notification_header.Len()) {
				FNotificationInfo info(FText::FromString(notification_header));
				info.FadeInDuration = 0.25f;
				info.FadeOutDuration = 0.25f;
				info.ExpireDuration = 30;
				info.SubText = FText::FromString(relative_path);
				info.ButtonDetails.Add(FNotificationButtonInfo(FText::FromString("Dismiss"), FText(), FSimpleDelegate::CreateLambda([notification_id]() {
					FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
					provider.set_notification_state(notification_id, SNotificationItem::CS_Success);
				})));
				auto notification_item = notification_manager.AddNotification(info);
				notification_item->SetCompletionState(SNotificationItem::CS_Pending);
				provider.notifications.Add( { notification_id, notification_item });
			}
		}
		break;
		default:
			ensure(false); // Completeness
			break;
	}
	provider.on_source_control_state_changed.Broadcast();
}

FArkSourceControlProvider::FArkSourceControlProvider() {
}

const FString& FArkSourceControlProvider::get_workspace_full_path() {
	return FArkSourceControlModule::Get().settings.workspace_path;
}

FString FArkSourceControlProvider::get_relative_path(const FString& absolute_path) {
	if (!absolute_path.StartsWith(get_workspace_full_path())) {
		return absolute_path;
	}
	return absolute_path.RightChop(get_workspace_full_path().Len());
}

FString FArkSourceControlProvider::get_absolute_path(const FString& relative_path) {
	return FString::Printf(TEXT("%s%s"), *get_workspace_full_path(), *relative_path);
}

TArray<FString> FArkSourceControlProvider::get_relative_paths(const TArray<FString>& files) {
	// Unfortunately here we can't just call USourceControlHelpers::AbsoluteFilenames
	// as when some instances we receive a relative path like ../../../etc (e.g. CheckOut) 
	// and all is fine, but some times we get Content/etc and then this whole things fails (e.g. CheckIn).
	// So we have our own way to handle this conversion...
	TArray<FString> relative_paths;
	relative_paths.Reserve(files.Num());
	for (int32 i = 0; i != files.Num(); ++i) {
		const FString& file = files[i];
		
		FString absolute_path;

		//FString relative_path;
		if (!FPaths::IsRelative(file)) {
			// It's absolute
			absolute_path = file;
		}
		else if (file.Len() > 1 && file[0] == '.' && (file[1] == '.' || file[1] == '/')) {
			// It's a relative path with . or .., so we need to convert to absolute and then trim
			absolute_path = FPaths::ConvertRelativePathToFull(file);
		}
		else {
			// Should be a proper relative path (like in the case of CheckIn)
			absolute_path = get_absolute_path(file);
		}

		// Ensure that when we're building the commands, relative paths have the trailing / for directories
		// since ark requires it, but UE has issues with it when displaying the changelist window
		FString relative_path = get_relative_path(absolute_path);
		if (relative_path.Len() && relative_path[relative_path.Len()-1] != '/') {
			auto* file_state_ref = relative_path_to_file_state_ref.Find(relative_path);
			if (!file_state_ref) {
				// Couldn't find it, so try with / to see if it's actually a directory
				FString relative_path_as_directory = FString::Printf(TEXT("%s/"), *relative_path);
				file_state_ref = relative_path_to_file_state_ref.Find(relative_path_as_directory);
				if (file_state_ref) {
					// It was a directory
					relative_path = relative_path_as_directory;
				}
				else {
					// If we couldn't find it either, just try to ask OS and pray it's not a deleted file!
					if (FPaths::DirectoryExists(FString::Printf(TEXT("%s/"), *absolute_path))) {
						relative_path = relative_path_as_directory;
					}
				}
			}
		}
		relative_paths.Emplace(relative_path);
	}
	return relative_paths;
}

bool FArkSourceControlProvider::is_initialized(const FString& path) {
	if (!path.Len()) {
		return false;
	}
	FString ark_folder_path;
	if (path[path.Len()-1] == '/' || path[path.Len()-1] == '\\') {
		ark_folder_path = path + ARK_FOLDER;
	} else {
		ark_folder_path = path + '/' + ARK_FOLDER;
	}
	return FPaths::DirectoryExists(ark_folder_path);
}

bool FArkSourceControlProvider::initialize_workspace(bool only_update_host_and_email /*= false*/) { // @todo Add Linux / Mac support
#if !PLATFORM_WINDOWS
	return false;
#endif
	
	auto& settings = FArkSourceControlModule::Get().settings;
	if (settings.email.IsEmpty()) {
		LOG.Error(FText::FromString(TEXT("No email set. Cannot initialize an Ark workspace without it.")));
		return false;
	}

	FString host = settings.host;

	if (host.IsEmpty()) {
		host = FString::Printf(TEXT("localhost:%s"), TEXT(ARK_DEFAULT_PORT));
	}
	
	FString init_command;
	if (only_update_host_and_email) {
		init_command = FString::Printf(TEXT("/c %s init -email %s -host %s"), *settings.binary_path, *settings.email, *host);
	}
	else {
		init_command = FString::Printf(TEXT("/c %s init -email %s -host %s -add_default_ignore %d -add_default_locks %d -built_in_server %d"), *settings.binary_path, *settings.email, *host, settings.add_default_ignore, settings.add_default_locks, settings.use_built_in_server);
	}
	FString std_out;
	FString std_err;
	int32 return_code;
	bool success = FPlatformProcess::ExecProcess(TEXT("cmd.exe"), *init_command, &return_code, &std_out, &std_err, *get_workspace_full_path(), true);

	if (std_out.Len()) {
		LOG.Info(FText::FromString(std_out));
	}
	if (std_err.Len()) {
		LOG.Error(FText::FromString(std_err));
	}
	
	if (!success || return_code != 0) {
		return false;
	}

	// Double check in case the executable chosen doesn't fail without telling us
	return is_initialized(settings.workspace_path);
}

bool FArkSourceControlProvider::launch_ark() {

	if (ark_proc_handle.IsValid() && FPlatformProcess::IsProcRunning(ark_proc_handle)) {
		return true;
	}

	auto& settings = FArkSourceControlModule::Get().settings;

	if (settings.binary_path.IsEmpty()) {
		LOG.Error(FText::FromString(TEXT("Ark binary path is not set")));
		return false;
	}
	if (!FPaths::FileExists(settings.binary_path)) {
		LOG.Error(FText::FromString(FString::Printf(TEXT("Ark binary path (%s) is not valid"), *settings.binary_path)));
		return false;
	}

	if (!is_initialized(settings.workspace_path)) {
		LOG.Info(FText::FromString(FString::Printf(TEXT("%s does not have an Ark workspace initialized. Initializing it for convenience."), *settings.workspace_path)));
		if (!initialize_workspace()) {
			LOG.Error(FText::FromString(TEXT("Failed to initialize Ark workspace. Are you sure you have the correct Ark binary path and workspace path?")));
			return false;
		}
	}

	FString host = settings.host;

	if (host.IsEmpty()) {
		host = FString::Printf(TEXT("localhost:%s"), TEXT(ARK_DEFAULT_PORT));
	}

	uint32 process_id;
	FString arguments = FString::Printf(TEXT("gui -workspace \"%s\" -host %s -email %s"), *get_workspace_full_path(), *host, *settings.email);
	LOG.Info(FText::FromString(FString::Printf(TEXT("Launching: %s %s"), *settings.binary_path, *arguments)));
	ark_proc_handle = FPlatformProcess::CreateProc(*settings.binary_path, *arguments, false, true, true, &process_id, 0, nullptr, nullptr, nullptr);

	return ark_proc_handle.IsValid();
}

bool FArkSourceControlProvider::connect_to_ark(uint32 process_id) {
	if (ark_proc_handle.IsValid() && FPlatformProcess::IsProcRunning(ark_proc_handle)) {
		return true;
	}

	ark_proc_handle = FPlatformProcess::OpenProcess((uint32)process_id);

	return ark_proc_handle.IsValid();
}

void FArkSourceControlProvider::process_command_sync(FArkSourceControlCommand* command) {
	bool finished = false;

	bool can_continue_executing;
	do {
		ark_plugin_c_update(plugin);
		finished = command->execute();
		// Originally we had here time check, but since we wanted to for example be able to commit
		// the project from inside Unreal Editor, we now abort if there's no more Ark process
		can_continue_executing = command->can_execute_without_ark_instance() || (ark_proc_handle.IsValid() && FPlatformProcess::IsProcRunning(ark_proc_handle));
	} while (!finished && can_continue_executing );

	if (!finished && !can_continue_executing) {
		command->abort();
	}

	on_source_control_state_changed.Broadcast();
}

void FArkSourceControlProvider::push_request(FArkSourceControlCommand* command, uint64_t request_id) {
	FArkSourceControlRequest request;
	request.request_id = request_id;
	request.command_id = command->id;
	requests.Add(request);
}

FSourceControlStateRef* FArkSourceControlProvider::get_file_state_ref(const FString& relative_path, bool* out_was_added /*= nullptr*/) {
	FSourceControlStateRef* file_state_ref = relative_path_to_file_state_ref.Find(relative_path);
	if (out_was_added) {
		*out_was_added = file_state_ref == nullptr;
	}
	if (!file_state_ref) {
		file_state_ref = &relative_path_to_file_state_ref.Add(relative_path, MakeShareable(new FArkSourceControlState(relative_path)));
	}

	return file_state_ref;
}

FArkSourceControlState* FArkSourceControlProvider::get_file_state(const FString& relative_path, bool* out_was_added /*= nullptr*/) {
	FSourceControlStateRef* file_state_ref = get_file_state_ref(relative_path, out_was_added);
	
	return (FArkSourceControlState*)file_state_ref->operator->();
}

FSourceControlChangelistStateRef FArkSourceControlProvider::get_ws_cl_or_add(int64 ws_cl_id) {
	FSourceControlChangelistStateRef* ws_cl_ptr = ws_cls.Find(ws_cl_id);
	if (!ws_cl_ptr) {
		ws_cl_ptr = &ws_cls.Add(ws_cl_id, MakeShareable(new FArkSourceControlChangelistState(FArkSourceControlChangelist(ws_cl_id))));
	}

	return *ws_cl_ptr;
}

void FArkSourceControlProvider::clear_all_ws_cls() {
	ws_cls.Empty();
}

void FArkSourceControlProvider::delete_ws_cl(int64_t ws_cl_id) {
	ws_cls.Remove(ws_cl_id);
}

void FArkSourceControlProvider::update_ws_cls(bool async /*= true*/) {
	ISourceControlProvider::Execute(ISourceControlOperation::Create<FUpdatePendingChangelistsStatus>(), async ? EConcurrency::Asynchronous : EConcurrency::Synchronous);
}

void FArkSourceControlProvider::force_update_relative_paths(const TArray<FString>& relative_paths) {
	ISourceControlProvider::Execute(ISourceControlOperation::Create<FUpdateStatus>(), relative_paths);
}

void FArkSourceControlProvider::force_update_current_directory_relative_paths() {
	FContentBrowserModule& content_browser_module = FModuleManager::Get().LoadModuleChecked<FContentBrowserModule>("ContentBrowser");
	FContentBrowserItemPath current_path = content_browser_module.Get().GetCurrentPath();
	if (current_path.HasInternalPath()) {
		FString relative_path;
		bool success = FPackageName::TryConvertGameRelativePackagePathToLocalPath(current_path.GetInternalPathString(), relative_path);
		if (success) {
			FString absolute_path = FPaths::ConvertRelativePathToFull(relative_path);
			TArray<FString> files;
			IPlatformFile::GetPlatformPhysical().FindFiles(files, *absolute_path, TEXT(""));
			if (files.Num()) {
				force_update_relative_paths(files);
			}
		}
	}
}

bool FArkSourceControlProvider::get_asset(const FString& relative_path, uint32 cl_id, uint32 cl_revision, FString& out_temp_file_absolute_path, EConcurrency::Type concurrency) {
	ECommandResult::Type result = ISourceControlProvider::Execute(ISourceControlOperation::Create<FGetFile>(FString::Printf(TEXT("%d"), cl_id), FString::Printf(TEXT("%d"), cl_revision), relative_path), concurrency);

	if (result == ECommandResult::Type::Succeeded) {
		FArkSourceControlState* file_state = get_file_state(relative_path);
		Ark_Plugin_String temp_relative_path = ark_plugin_c_string_talloc(plugin, TCHAR_TO_UTF8(*relative_path));
		Ark_Plugin_String temp_file_absolute_path = ark_plugin_c_get_temp_file_absolute_path_talloc(plugin, temp_relative_path, cl_id, cl_revision, file_state->file.asset_id);
		out_temp_file_absolute_path = ARK_PLUGIN_STRING_TO_FSTRING(temp_file_absolute_path);

		return true;
	}
	return false;
}

void FArkSourceControlProvider::show_notification_request_lock_ownership(const FString& relative_path, const FString& lock_owner) {
	
	FArkNotificationInfo info = create_notification_info(FString::Printf(TEXT("%s is locked"), *FPaths::GetPathLeaf(relative_path)), FString::Printf(TEXT("%s is currently locked by %s.\nWould you like to request lock ownership?"), *relative_path, *lock_owner));
	info.base_info.ButtonDetails.Add(FNotificationButtonInfo(FText::FromString("Request Lock"), FText::FromString("Request lock ownership to current owner"), FSimpleDelegate::CreateLambda([notification_id = info.notification_id, relative_path]() {
		FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
		ark_plugin_c_request_lock_ownership(provider.plugin, ark_plugin_c_string_talloc(provider.plugin, TCHAR_TO_UTF8(*relative_path)));
		provider.set_notification_state(notification_id, SNotificationItem::CS_None);
	})));
	info.base_info.ButtonDetails.Add(FNotificationButtonInfo(FText::FromString("Dismiss"), FText(), FSimpleDelegate::CreateLambda([notification_id = info.notification_id, relative_path]() {
		FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;
		provider.set_notification_state(notification_id, SNotificationItem::CS_None);
	})));
	auto notification_item = FSlateNotificationManager::Get().AddNotification(info.base_info);
	notification_item->SetCompletionState(SNotificationItem::CS_Pending);
	notifications.Add( { info.notification_id, notification_item } );
}

FArkNotificationInfo FArkSourceControlProvider::create_notification_info(const FString& text, const FString& sub_text) {
	last_ue_notification_id += 1;
	FArkNotificationInfo notification = { FNotificationInfo(FText::FromString(text)), last_ue_notification_id };
	notification.base_info.SubText = FText::FromString(sub_text);
	notification.base_info.FadeInDuration = 0.25f;
	notification.base_info.FadeOutDuration = 0.25f;
	notification.base_info.ExpireDuration = 30;
	notification.notification_id = last_ue_notification_id;
	return notification;
}

void FArkSourceControlProvider::set_notification_state(uint64_t notification_id, SNotificationItem::ECompletionState state) {
	for (int32 i = 0; i < notifications.Num(); ++i) {
		if (notifications[i].notification_id == notification_id) {
			auto& notification = notifications[i].notification;
			notification->SetCompletionState(state);
			notification->SetFadeOutDuration(2);
			notification->SetExpireDuration(0); // This will make it fade out
			notifications.RemoveAtSwap(i);
			break;
		}
	}
	ark_plugin_c_notification_dismiss(plugin, notification_id);
}


// ISourceControlProvider

void FArkSourceControlProvider::Init(bool bForceConnection) {
	ON_SCOPE_EXIT
	{
		if (bForceConnection) {
			ISourceControlProvider::Execute(ISourceControlOperation::Create<FConnect>(), EConcurrency::Type::Synchronous);
		}
	};
	
	if (ark_library_handle != nullptr) { return; } // Already initialized
	LOG.Info(FText::FromString("Initializing Ark Vcs Connection"));

	FString LibraryPath = FPaths::Combine(IPluginManager::Get().FindPlugin("ArkSourceControl")->GetBaseDir(),  "Source/ArkSourceControl/Ark_Plugin/ark_plugin.dll");
	
	ark_library_handle = FPlatformProcess::GetDllHandle(*LibraryPath);

	if (ark_library_handle) {
		plugin = ark_plugin_c_init(TCHAR_TO_UTF8(*get_workspace_full_path()),
								   TCHAR_TO_UTF8(*FPaths::ConvertRelativePathToFull(FPaths::DiffDir())),
								   &FArkSourceControlProvider::plugin_alloc, 
								   &FArkSourceControlProvider::plugin_free, 
								   &FArkSourceControlProvider::plugin_realloc, 
								   &FArkSourceControlProvider::plugin_logger,
								   &FArkSourceControlProvider::plugin_response,
								   &FArkSourceControlProvider::plugin_notification);

		if (plugin) {
			LOG.Info(FText::FromString("Initialized Ark Plugin C Binding"));
		} else {
			LOG.Error(FText::FromString("Failed to initialize Ark Plugin C Binding"));
		}
	} else {
		FMessageDialog::Open(EAppMsgType::Ok, LOCTEXT("ArkLibraryError", "Failed to load ark_plugin.dll. This means that connection to Ark Vcs is not going to work."));
	}
}

void FArkSourceControlProvider::Close()
{
	LOG.Info(FText::FromString("Closing Ark Vcs Connection"));
	if (ark_library_handle) {
		ark_plugin_c_deinit(plugin);
		plugin = nullptr;
		FPlatformProcess::FreeDllHandle(ark_library_handle);
		ark_library_handle = nullptr;
	}
}

FText FArkSourceControlProvider::GetStatusText() const {
	return status_text;
}

TMap<ISourceControlProvider::EStatus, FString> FArkSourceControlProvider::GetStatus() const {
	auto& Settings = FArkSourceControlModule::Get().settings;
	
	Ark_Plugin_Workspace_Info workspace_info = ark_plugin_c_workspace_info(plugin);

	TMap<EStatus, FString> Result;
	Result.Add(EStatus::Enabled, IsEnabled() ? TEXT("Yes") : TEXT("No") );
	Result.Add(EStatus::Connected, (IsEnabled() && IsAvailable()) ? TEXT("Yes") : TEXT("No") );
	Result.Add(EStatus::Repository, FPaths::ConvertRelativePathToFull(FPaths::ProfilingDir()));
	Result.Add(EStatus::Remote, Settings.host);
	Result.Add(EStatus::Branch, ARK_PLUGIN_STRING_TO_FSTRING(workspace_info.branch_str));
	Result.Add(EStatus::Email, Settings.email);
	return Result;
}

bool FArkSourceControlProvider::IsEnabled() const {
	return is_ark_running;
}

bool FArkSourceControlProvider::IsAvailable() const {
	return true;
}

ECommandResult::Type FArkSourceControlProvider::GetState(const TArray<FString>& InFiles, TArray<FSourceControlStateRef>& OutState, EStateCacheUsage::Type InStateCacheUsage) {
	TArray<FString> relative_paths = get_relative_paths(InFiles);

	if(InStateCacheUsage == EStateCacheUsage::ForceUpdate) {
		ISourceControlProvider::Execute(ISourceControlOperation::Create<FUpdateStatus>(), relative_paths);
	}
	
	for (const FString& relative_path : relative_paths)	{
		OutState.Add(static_cast<FSourceControlStateRef>(*get_file_state_ref(relative_path)));
	}
	return ECommandResult::Type::Succeeded;
}

ECommandResult::Type FArkSourceControlProvider::GetState(const TArray<FSourceControlChangelistRef>& InChangelists, TArray<FSourceControlChangelistStateRef>& OutState, EStateCacheUsage::Type InStateCacheUsage) {
	for (auto& ChangelistRef : InChangelists) {
		FSourceControlChangelistStateRef ChangelistStateRef = get_ws_cl_or_add(static_cast<FArkSourceControlChangelist&>(ChangelistRef.Get()).ws_cl_id);
		OutState.Add(ChangelistStateRef);
	}
	return ECommandResult::Type::Succeeded;
}

TArray<FSourceControlStateRef> FArkSourceControlProvider::GetCachedStateByPredicate(TFunctionRef<bool(const FSourceControlStateRef&)> Predicate) const {
	// @todo
	TArray<FSourceControlStateRef> CachedStates;
	CachedStates.Reserve(relative_path_to_file_state_ref.Num());
	for (auto& entry : relative_path_to_file_state_ref) {
		if (Predicate(entry.Value)) {
			CachedStates.Add(entry.Value);
		}
	}
	return CachedStates;
}

FDelegateHandle FArkSourceControlProvider::RegisterSourceControlStateChanged_Handle(const FSourceControlStateChanged::FDelegate& SourceControlStateChanged) {
	return on_source_control_state_changed.Add(SourceControlStateChanged);
}

void FArkSourceControlProvider::UnregisterSourceControlStateChanged_Handle(FDelegateHandle Handle) {
	on_source_control_state_changed.Remove(Handle);
}

ECommandResult::Type FArkSourceControlProvider::Execute(const FSourceControlOperationRef& InOperation, FSourceControlChangelistPtr InChangelist, const TArray<FString>& InFiles, EConcurrency::Type InConcurrency, const FSourceControlOperationComplete& InOperationCompleteDelegate) {
	FName operation_name = InOperation->GetName(); 
	FArkSourceControlCommand* command = nullptr;

	bool ignore_command = false;

	if (operation_name == "Connect") {
		command = new FArkSourceControlCommandConnect(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "UpdateStatus") {
		command = new FArkSourceControlCommandUpdateStatus(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "UpdateChangelistsStatus") {
		command = new FArkSourceControlCommandUpdateChangelistsStatus(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "CheckOut") {
		command = new FArkSourceControlCommandCheckOut(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "Revert") {
		command = new FArkSourceControlCommandRevert(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "CheckIn") {
		command = new FArkSourceControlCommandCheckIn(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "NewChangelist") {
		command = new FArkSourceControlCommandNewChangelist(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "DeleteChangelist") {
		command = new FArkSourceControlCommandDeleteChangelist(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "EditChangelist") {
		command = new FArkSourceControlCommandEditChangelist(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "MoveToChangelist") {
		command = new FArkSourceControlCommandMoveToChangelist(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "GetFile") {
		command = new FArkSourceControlCommandGetFile(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "Sync") {
		command = new FArkSourceControlCommandSync(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "Resolve") {
		command = new FArkSourceControlCommandResolve(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	} else if (operation_name == "MarkForAdd") {
		// Ark automatically adds files
		ignore_command = true;
	} else if (operation_name == "Delete") {
		command = new FArkSourceControlCommandDelete(InOperation, InChangelist, InOperationCompleteDelegate, InConcurrency, InFiles);
	}

	if (command) {
		
		#if ARK_PLUGIN_DEBUG
		LOG.Info(FText::FromString(FString::Printf(TEXT("Executing Operation: %s"), *operation_name.ToString())));
		#endif // ARK_PLUGIN_DEBUG

		commands.Add(command);
		if (InConcurrency == EConcurrency::Type::Synchronous) {
			process_command_sync(command);
			commands.Remove(command);
			delete command;
		}
		return ECommandResult::Succeeded;
	}

	if (ignore_command) {
		return ECommandResult::Succeeded;
	}

	LOG.Error(FText::FromString(FString::Printf(TEXT("Ark Plugin does not current support %s operation, ignoring"), *operation_name.ToString())));

	return ECommandResult::Failed;
}

bool FArkSourceControlProvider::CanExecuteOperation(const FSourceControlOperationRef& InOperation) const {
	if (InOperation->GetName() == "Shelve") {
		return false;
	}
	return true;
}

bool FArkSourceControlProvider::CanCancelOperation(const FSourceControlOperationRef& InOperation) const {
	// @todo Will get a pass at a later time
	return false;
}

void FArkSourceControlProvider::CancelOperation(const FSourceControlOperationRef& InOperation) {
}

TArray<TSharedRef<ISourceControlLabel>> FArkSourceControlProvider::GetLabels(const FString& InMatchingSpec) const {
	// Not supported
	TArray<TSharedRef<ISourceControlLabel>> labels;
	return labels;
}

TArray<FSourceControlChangelistRef> FArkSourceControlProvider::GetChangelists(EStateCacheUsage::Type InStateCacheUsage) {
	TArray<FSourceControlChangelistRef> changelists;
	for (auto& ws_cl_entry : ws_cls) {
		changelists.Add(MakeShared<FArkSourceControlChangelist, ESPMode::ThreadSafe>(static_cast<FArkSourceControlChangelistState&>(ws_cl_entry.Value.Get()).changelist));
	}
	return changelists;
}

TOptional<bool> FArkSourceControlProvider::IsAtLatestRevision() const {
	Ark_Plugin_Workspace_Info workspace_info = ark_plugin_c_workspace_info(plugin);
	return workspace_info.current_cl_id == workspace_info.head_cl_id;
}

TOptional<int> FArkSourceControlProvider::GetNumLocalChanges() const {
	// If set, it will allow the submit button to be on
	return TOptional<int>();
}

void FArkSourceControlProvider::Tick() {
	is_ark_running = ark_proc_handle.IsValid() && FPlatformProcess::IsProcRunning(ark_proc_handle);

	bool finished_executing_commands = false;

	if (plugin) {
		ark_plugin_c_update(plugin);
	}
	
	for (int32 i = 0; i < commands.Num(); ++i) {
		if (commands[i]->execute()) {
			finished_executing_commands = true;
			FMemory::Free(commands[i]);
			commands.RemoveAt(i);
			--i;
		}
	}
	
	if (finished_executing_commands) {
		on_source_control_state_changed.Broadcast();
	}
}

#if SOURCE_CONTROL_WITH_SLATE
TSharedRef<SWidget> FArkSourceControlProvider::MakeSettingsWidget() const {
	return SNew(SArkSourceControlSettings); 
}
#endif // SOURCE_CONTROL_WITH_SLATE

#undef LOG

#undef LOCTEXT_NAMESPACE
