﻿#include "SArkSourceControlSettings.h"

#include "ArkSourceControl.h"
#include "Widgets/Input/SFilePathPicker.h"
#include "Widgets/Input/SDirectoryPicker.h"
#include "EditorDirectories.h"

#define LOCTEXT_NAMESPACE "SArkSourceControlSettings"

void SArkSourceControlSettings::Construct(const FArguments& InArgs)
{
	const FText file_filter_type = NSLOCTEXT("ArkSourceControl", "Executables", "Executables");
#if PLATFORM_WINDOWS
	const FString file_filter_text = FString::Printf(TEXT("%s (*.exe)|*.exe"), *file_filter_type.ToString());
#else
	const FString file_filter_text = FString::Printf(TEXT("%s"), *file_filter_type.ToString());
#endif

	constexpr float padding = 2.0f;
	constexpr float fill_width = 2.0f;

	FText binary_path_tooltip = LOCTEXT("binary-path-tooltip", "Path to Ark binary");
	FText use_built_in_server_tooltip = LOCTEXT("use-built-in-server-tooltip", "Built-In Server means that the another instance of Ark will run for the Server automatically.\nThis is only useful if you want to work by yourself, otherwise it's better to have another machine that is always on running Ark Server.");
	FText add_default_ignore_tooltip = LOCTEXT("add-default-ignore-tooltip", "Add a default .ark_ignore file that should cover most of the generic ignores you'll need to add.\nIt's highly recommended you use this option if you're creating new project. You can always create or edit it later.");
	FText add_default_locks_tooltip = LOCTEXT("add-default-locks-tooltip", "Add a default .ark_locks file that should cover most of the generic binary file rules.\nIt's recommended you use this option if you're creating a new project. You can always create or edit it later.");

	ChildSlot
	[
		SNew(SVerticalBox)
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SHorizontalBox)
			+SHorizontalBox::Slot()
			[
				SNew(STextBlock)
				.Text(LOCTEXT("binary-path-label", "Ark Path"))
				.ToolTipText(binary_path_tooltip)
			]
			+SHorizontalBox::Slot()
			.FillWidth(fill_width)
			[
				SNew(SFilePathPicker)
				.BrowseButtonImage(FAppStyle::GetBrush("PropertyWindow.Button_Ellipsis"))
				.BrowseButtonStyle(FAppStyle::Get(), "HoverHintOnly")
				.BrowseButtonToolTip(binary_path_tooltip)
				.BrowseDirectory(FEditorDirectories::Get().GetLastDirectory(ELastDirectory::GENERIC_OPEN))
				.BrowseTitle(LOCTEXT("binary-path-browse-title", "File picker..."))
				.FilePath_Lambda([]() -> FString
				{
					return FArkSourceControlModule::Get().settings.binary_path;
				})
				.FileTypeFilter(file_filter_text)
				.OnPathPicked_Lambda([](const FString& new_binary_path)
				{
					FArkSourceControlModule::Get().settings.binary_path = new_binary_path.IsEmpty() ? "" : FPaths::ConvertRelativePathToFull(new_binary_path);
				})
			]
		]
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SHorizontalBox)
			.Visibility_Lambda([]()->EVisibility
			{
				return FArkSourceControlModule::Get().settings.use_built_in_server ? EVisibility::Collapsed : EVisibility::Visible;
			})
			+SHorizontalBox::Slot()
			[
				SNew(STextBlock)
				.Text(LOCTEXT("host-label", "Host"))
			]
			+SHorizontalBox::Slot()
			.FillWidth(fill_width)
			[
				SNew(SEditableTextBox)
				.HintText(FText::FromString("localhost:9000"))
				.Text_Lambda([] () -> FText
				{
					return FText::FromString(FArkSourceControlModule::Get().settings.host);
				})
				.OnTextChanged_Lambda([](const FText& new_host)
				{
					FArkSourceControlModule::Get().settings.host = new_host.ToString();
				})
			]
		]
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SHorizontalBox)
			.Visibility_Lambda([]()->EVisibility
			{
				return FArkSourceControlModule::Get().settings.use_built_in_server ? EVisibility::Visible : EVisibility::Collapsed;
			})
			+SHorizontalBox::Slot()
			[
				SNew(STextBlock)
				.Text(LOCTEXT("port-label", "Port"))
			]
			+SHorizontalBox::Slot()
			.FillWidth(fill_width)
			[
				SNew(SEditableTextBox)
				.HintText(FText::FromString(ARK_DEFAULT_PORT))
				.Text_Lambda([] () -> FText
				{
					return FText::FromString(FArkSourceControlModule::Get().settings.get_port());
				})
				.OnTextChanged_Lambda([](const FText& new_port)
				{
					FArkSourceControlModule::Get().settings.set_localhost_with_port(new_port.ToString());
				})
			]
		]
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SHorizontalBox)
			+SHorizontalBox::Slot()
			[
				SNew(STextBlock)
				.Text(LOCTEXT("email-label", "Email"))
			]
			+SHorizontalBox::Slot()
			.FillWidth(fill_width)
			[
				SNew(SEditableTextBox)
				.Text_Lambda([] () -> FText
				{
					return FText::FromString(FArkSourceControlModule::Get().settings.email);
				})
				.OnTextChanged_Lambda([](const FText& new_email)
				{
					FArkSourceControlModule::Get().settings.email = new_email.ToString();
				})
			]
		]
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SHorizontalBox)
			+SHorizontalBox::Slot()
			[
				SNew(STextBlock)
				.Text(LOCTEXT("workspace-label", "Workspace"))
			]
			+SHorizontalBox::Slot()
			.FillWidth(fill_width)
			[
				SNew(SDirectoryPicker)
				.Directory(FArkSourceControlModule::Get().settings.workspace_path)
				.OnDirectoryChanged_Lambda([](const FString& new_workspace_path)
				{
					FArkSourceControlModule::Get().settings.workspace_path = FPaths::ConvertRelativePathToFull(new_workspace_path);
				})
			]
		]
		+SVerticalBox::Slot()
		.AutoHeight()
		.Padding(padding)
		[
			SNew(SVerticalBox)
			.Visibility_Lambda([]()->EVisibility{
				return FArkSourceControlModule::Get().provider.is_initialized(FArkSourceControlModule::Get().settings.workspace_path) ? EVisibility::Collapsed : EVisibility::Visible;
			})
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(SSeparator)
			]
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(STextBlock)
				.Text(LOCTEXT("not-initialized-label", "Current Project does not have an Ark Workspace. To initialize one, use the button bellow."))
			]
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(SHorizontalBox)
				+SHorizontalBox::Slot()
				[
					SNew(STextBlock)
					.Text(LOCTEXT("use-built-in-server-label", "Use Built-In Server"))
					.ToolTipText(use_built_in_server_tooltip)
				]
				+SHorizontalBox::Slot()
				.FillWidth(fill_width)
				[
					SNew(SCheckBox)
					.IsChecked(FArkSourceControlModule::Get().settings.use_built_in_server ? ECheckBoxState::Checked : ECheckBoxState::Unchecked)
					.ToolTipText(use_built_in_server_tooltip)
					.OnCheckStateChanged_Lambda([](ECheckBoxState new_state)
					{
						FArkSourceControlModule::Get().settings.use_built_in_server = new_state == ECheckBoxState::Checked;
					})
				]
			]
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(SHorizontalBox)
				+SHorizontalBox::Slot()
				[
					SNew(STextBlock)
					.Text(LOCTEXT("add-default-ignore-label", "Add Default Ignore File"))
					.ToolTipText(add_default_ignore_tooltip)
				]
				+SHorizontalBox::Slot()
				.FillWidth(fill_width)
				[
					SNew(SCheckBox)
					.IsChecked(FArkSourceControlModule::Get().settings.add_default_ignore ? ECheckBoxState::Checked : ECheckBoxState::Unchecked)
					.ToolTipText(add_default_ignore_tooltip)
					.OnCheckStateChanged_Lambda([](ECheckBoxState new_state)
					{
						FArkSourceControlModule::Get().settings.add_default_ignore = new_state == ECheckBoxState::Checked;
					})
				]
			]
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(SHorizontalBox)
				+SHorizontalBox::Slot()
				[
					SNew(STextBlock)
					.Text(LOCTEXT("add-default-locks-label", "Add Default Locks File"))
					.ToolTipText(add_default_locks_tooltip)
				]
				+SHorizontalBox::Slot()
				.FillWidth(fill_width)
				[
					SNew(SCheckBox)
					.IsChecked(FArkSourceControlModule::Get().settings.add_default_locks ? ECheckBoxState::Checked : ECheckBoxState::Unchecked)
					.ToolTipText(add_default_locks_tooltip)
					.OnCheckStateChanged_Lambda([](ECheckBoxState new_state)
					{
						FArkSourceControlModule::Get().settings.add_default_locks = new_state == ECheckBoxState::Checked;
					})
				]
			]
			+SVerticalBox::Slot()
			.AutoHeight()
			.Padding(padding)
			[
				SNew(SHorizontalBox)
				+SHorizontalBox::Slot()
				.FillWidth(1.0f)
				[
					// @todo Support Linux / Mac
#if PLATFORM_WINDOWS
					SNew(SButton)
					.Text(LOCTEXT("initialize-workspace-label", "Initialize Ark Workspace"))
					.HAlign(HAlign_Center)
					.ContentPadding(6.0f)
					.OnClicked_Lambda([]() -> FReply
					{
						FArkSourceControlModule::Get().provider.initialize_workspace();
						return FReply::Handled();
					})
#else
					SNew(STextBlock)
					.Text(LOCTEXT("initialize-workspace-not-supported-label", "Cannot initialize Ark Workspace on this plaftorm at this time. Please initialize it via Ark directly."))
#endif
				]
			]
		]
	];
}

SArkSourceControlSettings::~SArkSourceControlSettings()
{
	FArkSourceControlModule::Get().settings.save_settings();
}

#undef LOCTEXT_NAMESPACE
