﻿#include "FArkSourceControlState.h"

#include "RevisionControlStyle/RevisionControlStyle.h"
#include "Templates/SharedPointer.h"

void FArkSourceControlState::update(const Ark_Plugin_File& in_file) {

	resolve_info = {};

	file = in_file;
	if (file.lock_owner.count) {
		lock_owner = ARK_PLUGIN_STRING_TO_FSTRING(file.lock_owner);
	}
	else {
		lock_owner.Reset();
	}
	// Clear it since we store it in the string above
	file.lock_owner.data = nullptr;
	file.lock_owner.count = 0;

	if (file.conflict.state != 0) {
		resolve_info.BaseFile = relative_path;
		resolve_info.RemoteFile = relative_path;
		// They will be filled upon receiving history
		resolve_info.BaseRevision = "";
		resolve_info.RemoteRevision = "";
	}
}

void FArkSourceControlState::update(const Ark_Plugin_File_History& in_history) {
	revisions.Reset(0);
	revisions.Reserve(in_history.revisions.count);
	// Add them in reversed order so lastest are at the top
	for (int32 i = in_history.revisions.count-1; i >= 0; --i) {
		TSharedRef<FArkSourceControlRevision, ESPMode::ThreadSafe> revision = MakeShareable(new FArkSourceControlRevision(absolute_path, *(in_history.revisions.data + i), i + 1));
		revisions.Add(revision);

		if (file.conflict.state != 0) {
			if (file.conflict.remote_cl_id == revision->cl_id && file.conflict.remote_cl_revision == revision->cl_revision) {
				resolve_info.RemoteRevision = revision->revision;
			}
			else if (file.conflict.base_cl_id == revision->cl_id && file.conflict.base_cl_revision == revision->cl_revision) {
				resolve_info.BaseRevision = revision->revision;
			}
		}
	}
}

FName FArkSourceControlState::get_icon_name() const {
	// RevisionControl.OpenForAdd                            // Green +
	// RevisionControl.MarkedForDelete                       // Red -
	// RevisionControl.NotAtHeadRevision                     // Yellow exclamation mark
	// RevisionControl.NotInDepot                            // Yellow question mark
	// RevisionControl.ConflictedState                       // 2 arrows colliding
	// RevisionControl.Branched                              // Mirrored tilted h
	// RevisionControl.CheckedOutByOtherUser                 // Red person
	// RevisionControl.CheckedOutByOtherUserBadge            // Blue check
	// RevisionControl.CheckedOutByOtherUserOtherBranch      // Green wave
	// RevisionControl.CheckedOutByOtherUserOtherBranchBadge // Yellow check
	// RevisionControl.ModifiedOtherBranch                   // Nonsense
	// RevisionControl.ModifiedBadge                         // Yellow exclamation mark
	// RevisionControl.ModifiedLocally                       // Grey check with dot
	// RevisionControl.Locked                                // Closed lock
	// RevisionControl.Unlocked                              // Open lock
	// RevisionControl.StatusBar.AtLatestRevision            // Check with line above
	// RevisionControl.StatusBar.NotAtLatestRevision         // Orange down wrrow
	// RevisionControl.StatusBar.Promote                     // Arrow going around and up
	// RevisionControl.StatusBar.NoLocalChanges              // Grey check with line bellow
	// RevisionControl.StatusBar.HasLocalChanges             // Blue arrow up
	// RevisionControl.StatusBar.Conflicted                  // Yellow arrows colliding (bigger)
	
	// If we're adding, then we just put + there
	if (file.status == File_Status_ADD) {
		return "RevisionControl.OpenForAdd";
	}

	// With other cases, the most important think is knowing we're not up to date
	bool is_current = IsCurrent();

	if (file.conflict.state == File_Conflict_NEEDS_RESOLVING) {
		return "RevisionControl.StatusBar.Conflicted";
	}

	if (file.status == File_Status_MOD) {
		if (!is_current) {
			return "RevisionControl.CheckedOutByOtherUserOtherBranchBadge";
		}
		if (file.lock & File_Lock_LOCK_OWNER) {
			return "RevisionControl.CheckedOut";
		}
		if (file.lock & File_Lock_LOCK_REMOTE) {
			return "RevisionControl.CheckedOutByOtherUserBadge";
		}
		if (file.lock & File_Lock_LOCK_LOCAL) {
			return "RevisionControl.StatusBar.NoLocalChanges";
		}
		return "RevisionControl.CheckedOutByOtherUserBadge";
	}

	// For all the other cases, if it's not current we want to show yellow exclamation mark
	if (!is_current) {
		return "RevisionControl.NotAtHeadRevision";
	}

	if (file.status == File_Status_DEL) {
		return "RevisionControl.MarkedForDelete";
	}
	if (file.lock & File_Lock_LOCK_OWNER) {
		return "RevisionControl.Locked";
	}
	if (file.lock & File_Lock_LOCK_REMOTE) {
		return "RevisionControl.ModifiedLocally";
	}

	return NAME_None;
}


// ISourceControlState

int32 FArkSourceControlState::GetHistorySize() const {
	return revisions.Num();
}

TSharedPtr<ISourceControlRevision, ESPMode::ThreadSafe> FArkSourceControlState::GetHistoryItem(int32 HistoryIndex) const {
	return revisions[HistoryIndex];
}

TSharedPtr<ISourceControlRevision, ESPMode::ThreadSafe> FArkSourceControlState::FindHistoryRevision(int32 RevisionNumber) const {
	return revisions[RevisionNumber-1];
}

TSharedPtr<ISourceControlRevision, ESPMode::ThreadSafe> FArkSourceControlState::FindHistoryRevision(const FString& InRevision) const {
	// In theory we can just parse string, remove # and convert the text to a number and from there get the index (although remember it's inverted, so highest is at 0 for display purposes)
	for (int32 i = 0; i != revisions.Num(); ++i) {
		FArkSourceControlRevision& revision = (FArkSourceControlRevision&)revisions[i].Get();
		if (revision.revision == InRevision) {
			return revisions[i];
		}
	}
	return nullptr;
}

TSharedPtr<ISourceControlRevision, ESPMode::ThreadSafe> FArkSourceControlState::GetCurrentRevision() const {
	if (revisions.Num()) {
		return revisions[revisions.Num()-1];
	}
	return nullptr;
}

ISourceControlState::FResolveInfo FArkSourceControlState::GetResolveInfo() const {
	return resolve_info;
}

FSlateIcon FArkSourceControlState::GetIcon() const {
	FName icon_name = get_icon_name();
	if (!icon_name.IsNone()) {
		return FSlateIcon(FRevisionControlStyleManager::GetStyleSetName(), icon_name);
	}
	return FSlateIcon();
}

FText FArkSourceControlState::GetDisplayName() const {
	return FText::FromString(relative_path);
}

FText FArkSourceControlState::GetDisplayTooltip() const {
	TStringBuilderWithBuffer<TCHAR, 128> StringBuilder;
	
	if (!IsCurrent()) {
		Ark_Plugin_Workspace_Info workspace_info = ark_plugin_c_workspace_info(FArkSourceControlModule::Get().provider.plugin);
		StringBuilder.Appendf(TEXT("Outdated: latest %d (current %d)"), file.head_cl_id, workspace_info.current_cl_id);
	} else {
		StringBuilder.Append(TEXT("Up to date"));
	}

	if (file.conflict.state == File_Conflict_RESOLVED) {
		StringBuilder.Append(TEXT("\nConflicts Merged"));
	} else if (file.conflict.state == File_Conflict_NEEDS_RESOLVING) {
		StringBuilder.Append(TEXT("\nHas Conflicts"));
	}

	if (file.status != 0) {
		StringBuilder.Append(TEXT("\nHas local changes"));
	} else {
		StringBuilder.Append(TEXT("\nNo local changes"));
	}

	if (lock_owner.IsEmpty()) {
		StringBuilder.Append(TEXT("\nLock Ownership: n/a"));
	} else {
		StringBuilder.Appendf(TEXT("\nLock Ownership: %s"), *lock_owner);
	}
	
	return FText::FromString(StringBuilder.ToString());
}

TOptional<FText> FArkSourceControlState::GetWarningText() const {
	TOptional<FText> WarningText = ISourceControlState::GetWarningText();
	// We want to also ensure that we want to show tooltip when there's no lock by anybody but we actually have local changes
	if (!WarningText.IsSet() && file.status != File_Status_NONE && file.lock == File_Lock_NONE) {
		WarningText.Emplace(GetDisplayTooltip());
	}
	return WarningText;
}

const FString& FArkSourceControlState::GetFilename() const {
	return absolute_path;
}

const FDateTime& FArkSourceControlState::GetTimeStamp() const {
	return date_time;
}

bool FArkSourceControlState::CanCheckIn() const {
	return file.status != File_Status_NONE;
}

bool FArkSourceControlState::CanCheckout() const {
	return file.asset_id != 0 && ((file.lock & File_Lock_LOCK_OWNER) == 0);
}

bool FArkSourceControlState::IsCheckedOut() const {
	// Once you go down the check out path, it basically follows what perforce does which is annoying.
	// So in Ark's case, you can only lock when assets are already committed, so we fake being checked out
	// when assets are also added, otherwise UE doesn't save the files (and forces you to choose "Make Writable"...
	return (file.lock & File_Lock_LOCK_LOCAL) != 0;// || (file.status == File_Status_ADD) || (file.asset_id == 0);
}

bool FArkSourceControlState::IsCheckedOutOther(FString* Who) const {
	if (Who && file.lock != 0) {
		*Who = lock_owner;
	}
	return (file.lock & File_Lock_LOCK_REMOTE) != 0;
}

bool FArkSourceControlState::IsCurrent() const {
	Ark_Plugin_Workspace_Info workspace_info = ark_plugin_c_workspace_info(FArkSourceControlModule::Get().provider.plugin);
	if (workspace_info.current_cl_id >= file.head_cl_id) {
		return true;
	}

	for (int32 i = 0; i < workspace_info.committed_cl_ids.count; ++i) {
		if (*(workspace_info.committed_cl_ids.data + i) == file.head_cl_id) {
			return true;
		}
	}
	return false;
}

bool FArkSourceControlState::IsSourceControlled() const {
	return file.asset_id != 0;
}

bool FArkSourceControlState::IsAdded() const {
	return file.status == File_Status_ADD;
}

bool FArkSourceControlState::IsDeleted() const {
	return file.status == File_Status_DEL;
}

bool FArkSourceControlState::IsIgnored() const {
	return file.status == File_Status_IGNORED;
}

bool FArkSourceControlState::CanEdit() const {
	return true;
}

bool FArkSourceControlState::IsUnknown() const {
	return false;
}

bool FArkSourceControlState::IsModified() const {
	// IsModified doesn't mean the same as in Ark. In Ark, a file being modified
	// means that you still have it and it has changes when compared to "latest".
	// But in UE, you need to flag them as modified if they are added, modified or deleted,
	// otherwise when you try to commit it does a "revert unchanged" and uses this function
	// to know if something has changes or not...
	return file.status >= File_Status_ADD && file.status <= File_Status_DEL;
}

bool FArkSourceControlState::CanAdd() const {
	// CanAdd doesn't matter to us since we don't care about MarkForAdd. Ark automatically detects and marks things to be added.
	return false;
}

bool FArkSourceControlState::CanDelete() const {
	return true;
}

bool FArkSourceControlState::IsConflicted() const {
	return file.conflict.state != File_Conflict_NONE;
}

bool FArkSourceControlState::CanRevert() const {
	return file.status != File_Status_NONE || IsCheckedOut();
}
