#pragma once

#ifndef ARK_PLUGN_H
#define ARK_PLUGIN_H

#if defined _WIN32 || defined _WIN64
#define ARK_PLUGIN_API __declspec(dllimport)
#elif defined __linux__
#define ARK_PLUGIN_API __attribute__((visibility("default")))
#else
#define ARK_PLUGIN_API
#endif

#define ARK_PLUGIN_MSG_MAX_SIZE 65480

#define ARK_ARRAY(ARRAY_TYPE, ELEMENT_TYPE) struct ARRAY_TYPE { int64_t count = 0; ELEMENT_TYPE* data = NULL; }

#define ARK_FOLDER ".ark/"
#define ARK_BUILTIN_SERVER_FOLDER ".ark_server/"
#define ARK_IGNORE ".ark_ignore"
#define ARK_DEFAULT_PORT "9000"

typedef uint32_t Ark_Asset_Id;
typedef uint32_t Ark_Changelist_Id;
typedef uint32_t Ark_Revision;
typedef uint32_t Ark_Branch_Id;
typedef uint32_t Ark_Project_Id;
typedef uint8_t  Ark_Bool;

typedef uint8_t Ark_Plugin_File_Status;
enum Ark_Plugin_File_Status_Values {
	File_Status_NONE    = 0,
	File_Status_ADD     = 1,
	File_Status_MOD     = 2,
	File_Status_DEL     = 3,
	File_Status_IGNORED = 4,
};

typedef uint8_t Ark_Plugin_File_Lock; // Flags
enum Ark_Plugin_File_Lock_Values {
	File_Lock_NONE        = 0,
	File_Lock_LOCK_LOCAL  = 1,
	File_Lock_LOCK_REMOTE = 2,
	File_Lock_LOCK_OWNER  = 4,
};

typedef uint8_t Ark_Plugin_File_Conflict_State;
enum Ark_Plugin_File_Conflict_State_Values {
	File_Conflict_NONE            = 0,
	File_Conflict_NEEDS_RESOLVING = 1,
	File_Conflict_RESOLVED        = 2,
};

struct Ark_Plugin_File_Conflict {
	Ark_Plugin_File_Conflict_State state;
	Ark_Changelist_Id              base_cl_id;
	Ark_Revision                   base_cl_revision;
	Ark_Changelist_Id              remote_cl_id;
	Ark_Revision                   remote_cl_revision;
};

struct Ark_Plugin_String {
	int64_t  count = 0;
	uint8_t* data  = NULL;
};
ARK_ARRAY(Ark_Plugin_Array_String, Ark_Plugin_String);
ARK_ARRAY(Ark_Plugin_Array_U8,     uint8_t);
ARK_ARRAY(Ark_Plugin_Array_U32,    uint32_t);

struct Ark_Plugin_File {
	Ark_Asset_Id             asset_id   = 0;
	Ark_Changelist_Id        head_cl_id = 0;
	Ark_Plugin_String        lock_owner;
	Ark_Plugin_File_Status   status     = 0;
	Ark_Plugin_File_Lock     lock       = 0;
	Ark_Plugin_File_Conflict conflict   = 0;
};

ARK_ARRAY(Ark_Plugin_Array_File, Ark_Plugin_File);

struct Ark_Plugin_Ws_Cl {
	int64_t           ws_cl_id = 0; 
	Ark_Changelist_Id cl_id    = 0;
	Ark_Plugin_String comment;
	Ark_Plugin_Array_String relative_paths;
	Ark_Bool is_default = false;
};

ARK_ARRAY(Ark_Plugin_Array_Ws_Cl, Ark_Plugin_Ws_Cl);

struct Ark_Plugin_File_Changed {
	Ark_Plugin_String relative_path;
	Ark_Plugin_File   file;
};

ARK_ARRAY(Ark_Plugin_Array_File_Changed, Ark_Plugin_File_Changed);

struct Ark_Plugin_File_Get {
	Ark_Changelist_Id cl_id       = 0;
	Ark_Revision      cl_revision = 0;
	Ark_Asset_Id      asset_id    = 0;
};

ARK_ARRAY(Ark_Plugin_Array_File_Get, Ark_Plugin_File_Get);

typedef uint8_t Ark_Plugin_Request_Type;
enum Ark_Plugin_Request_Type_Values {
	Request_Type_PING              =  1,

	Request_Type_WS_CL_LIST        = 10,
	Request_Type_WS_CL_CREATE      = 11,
	Request_Type_WS_CL_DELETE      = 12,
	Request_Type_WS_CL_EDIT        = 13,
	Request_Type_WS_CL_MOVE_FILES  = 14,
	Request_Type_WS_CL_COMMIT      = 15,

	Request_Type_FILE_STATE        = 20,
	Request_Type_FILE_LOCK         = 21,
	Request_Type_FILE_UNLOCK       = 22,
	Request_Type_FILE_REVERT       = 23,
	Request_Type_FILE_RESOLVE      = 24,
	Request_Type_FILE_COMMIT       = 25,
	Request_Type_FILE_HISTORY      = 26,
	Request_Type_FILE_GET          = 27,

	Request_Type_LOCK_GIVE         = 40,
	Request_Type_LOCK_ASK          = 41,

	Request_Type_CL_GET            = 60,
};

struct Ark_Plugin_Time {
	uint64_t low  = 0;
    int64_t  high = 0;
};

struct Ark_Plugin_Calendar {
	int32_t year        = 0;
	int8_t  month       = 0;
	int8_t  day         = 0;
	int8_t  hour        = 0;
	int8_t  minute      = 0;
	int8_t  second      = 0;
	int16_t millisecond = 0;
};

struct Ark_PLugin_File_Revision {
	Ark_Changelist_Id cl_id       = 0;
	Ark_Revision      cl_revision = 0;
	int64_t           size        = 0;
	Ark_Plugin_Time   time;
	Ark_Plugin_String user;
	Ark_Plugin_String cl_comment;
	Ark_Plugin_String branch;
	Ark_Plugin_File_Status file_status = File_Status_NONE;
};

ARK_ARRAY(Ark_Plugin_Array_File_Revision, Ark_PLugin_File_Revision);

struct Ark_Plugin_File_History {
	Ark_Plugin_Array_File_Revision revisions;
};

ARK_ARRAY(Ark_Plugin_Array_File_History, Ark_Plugin_File_History);

struct Ark_Plugin_Response {
	uint64_t                request_id = 0;
	Ark_Plugin_Request_Type type  = 0;
	Ark_Bool                error = 0;
	union {
		struct {
			int64_t process_id = 0;
		} ping;
		struct {
			Ark_Plugin_Array_Ws_Cl ws_cls;
		} ws_cl_list;
		Ark_Plugin_Ws_Cl ws_cl_create;
		struct {} ws_cl_delete;
		struct {} ws_cl_edit;
		struct {} ws_cl_move_files;
		struct {} ws_cl_commit;
		struct {
			Ark_Plugin_Array_File files;
		} file_state;
		struct {
			Ark_Plugin_Array_File files;
		} file_lock;
		struct {
			Ark_Plugin_Array_File files;
		} file_unlock;
		struct {
			Ark_Plugin_Array_File files;
		} file_revert;
		struct {
		} file_resolve;
		struct {} file_commit;
		struct {
			Ark_Plugin_Array_File_History histories;
		} file_history;
		struct {
			Ark_Plugin_Array_U8 success;
		};
		struct {} lock_give;
		struct {} cl_get;
	};
};


typedef uint8_t Ark_Plugin_Notification_Type;
enum Ark_Plugin_Notification_Type_Values {
	Notification_Type_WORKSPACE_INFO =  1,
	Notification_Type_FILES_CHANGED  = 10,
	Notification_Type_LOCK_REQUEST   = 20,
	Notification_Type_LOCK_RESPONSE  = 21,
};

struct Ark_Plugin_Workspace_Info {
	Ark_Project_Id    project_id    = 0;
	Ark_Branch_Id     branch_id     = 0;
	Ark_Changelist_Id head_cl_id    = 0;
	Ark_Changelist_Id current_cl_id = 0;
	Ark_Plugin_Array_U32 committed_cl_ids;
	Ark_Plugin_String branch_str;   // <project name>:<branch name>
};

struct Ark_Plugin_Notification {
	Ark_Plugin_Notification_Type type = 0;
	int64_t id = 0; // If > 0, then it's an actionable notification we can dismiss
	union {
		Ark_Plugin_Workspace_Info     workspace_info;
		Ark_Plugin_Array_File_Changed files_changed;
		struct {
			Ark_Plugin_String relative_path;
			Ark_Plugin_String requester; 
		} lock_request;
		struct {
			Ark_Plugin_String relative_path;
			Ark_Plugin_String responder; 
			Ark_Bool          accepted = 0;
		} lock_response;
	};
};


typedef uint8_t Ark_Plugin_Data_Type;

enum Ark_Plugin_Data_Type_Values {
	Data_Type_REQUEST      = 0,
	Data_Type_RESPONSE     = 1,
	Data_Type_NOTIFICATION = 2,
};

struct Ark_Plugin_Data {
	uint64_t id = 0;     // Id of this message
	int64_t  count = 0;  // How big it is the data we're currently handling
	int64_t  cursor = 0; // Where we're at currently, if < count then we're still sending / receiving
	uint8_t* bytes = NULL;
	Ark_Plugin_Data_Type type = 0;
};

typedef void* (*ark_plugin_c_alloc_proc)   (int64_t);
typedef void  (*ark_plugin_c_free_proc)    (void*);
typedef void* (*ark_plugin_c_realloc_proc) (void*, int64_t, int64_t);
typedef void  (*ark_plugin_c_logger_proc)  (Ark_Plugin_String message, uint8_t level);

typedef void (*ark_plugin_c_response_callback)     (Ark_Plugin_Response*);
typedef void (*ark_plugin_c_notification_callback) (Ark_Plugin_Notification*);

struct Ark_Plugin {};

enum Ark_Plugin_Log_Level {
	Log_Level_DEBUG = 0,
	Log_Level_INFO  = 1,
	Log_Level_WARN  = 2,
	Log_Level_ERROR = 3,
};

extern "C" {
	extern ARK_PLUGIN_API Ark_Plugin* ark_plugin_c_init(char* workspace_path,
														   char* temp_path_override,
														   ark_plugin_c_alloc_proc alloc_proc, 
														   ark_plugin_c_free_proc free_proc, 
														   ark_plugin_c_realloc_proc realloc_proc, 
														   ark_plugin_c_logger_proc logger_proc, 
														   ark_plugin_c_response_callback response_callback, 
														   ark_plugin_c_notification_callback notification_callback);
	extern ARK_PLUGIN_API void ark_plugin_c_deinit     (Ark_Plugin* plugin);
	extern ARK_PLUGIN_API void ark_plugin_c_update     (Ark_Plugin* plugin);

	extern ARK_PLUGIN_API int8_t ark_plugin_c_is_connected(Ark_Plugin* plugin);

	extern ARK_PLUGIN_API Ark_Plugin_Workspace_Info ark_plugin_c_workspace_info(Ark_Plugin* plugin);

	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ping            (Ark_Plugin* plugin);

	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_list      (Ark_Plugin* plugin);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_create    (Ark_Plugin* plugin, Ark_Plugin_String comment, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_delete    (Ark_Plugin* plugin, int64_t ws_cl_id);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_edit      (Ark_Plugin* plugin, int64_t ws_cl_id, Ark_Plugin_String comment);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_move_files(Ark_Plugin* plugin, int64_t ws_cl_id, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_ws_cl_commit    (Ark_Plugin* plugin, Ark_Plugin_String comment, int64_t ws_cl_id, uint8_t keep_locks);

	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_state   (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_lock    (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_unlock  (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_revert  (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_resolve (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_commit  (Ark_Plugin* plugin, Ark_Plugin_String comment, Ark_Plugin_Array_String relative_paths, uint8_t keep_locks);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_history (Ark_Plugin* plugin, Ark_Plugin_Array_String relative_paths);

	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_file_get      (Ark_Plugin* plugin, Ark_Plugin_Array_File_Get files);


	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_lock_give     (Ark_Plugin* plugin, Ark_Plugin_String relative_path, Ark_Plugin_String user, uint8_t give);
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_lock_ownership(Ark_Plugin* plugin, Ark_Plugin_String relative_path);
	
	extern ARK_PLUGIN_API uint64_t ark_plugin_c_request_get_cl        (Ark_Plugin* plugin, uint32_t cl_id);

	extern ARK_PLUGIN_API void     ark_plugin_c_notification_dismiss  (Ark_Plugin* plugin, uint64_t notification_id);

	extern ARK_PLUGIN_API Ark_Plugin_String       ark_plugin_c_string_talloc      (Ark_Plugin* plugin, char* str);
	extern ARK_PLUGIN_API Ark_Plugin_Array_String ark_plugin_c_array_string_talloc(Ark_Plugin* plugin, int64_t size);
	extern ARK_PLUGIN_API void                    ark_plugin_c_array_string_set   (Ark_Plugin_Array_String* array, int64_t index, Ark_Plugin_String str);
	extern ARK_PLUGIN_API Ark_Plugin_String       ark_plugin_c_array_string_get   (Ark_Plugin_Array_String* array, int64_t index);
	extern ARK_PLUGIN_API Ark_Plugin_Calendar     ark_plugin_c_time_to_calendar   (Ark_Plugin* plugin, Ark_Plugin_Time time);

	extern ARK_PLUGIN_API Ark_Plugin_String ark_plugin_c_get_temp_file_absolute_path_talloc(Ark_Plugin* plugin, Ark_Plugin_String relative_path, uint32_t cl_id, uint32_t cl_revision, uint32_t asset_id);
}

#endif // ARK_PLUGIN_H
