﻿#include "FArkSourceControlCommand.h"

#include "ArkSourceControl.h"
#include "FArkSourceControlState.h"
#include "FArkSourceControlChangelistState.h"
#include "SourceControlHelpers.h"
#include "SourceControlOperations.h"

uint64_t FArkSourceControlCommand::last_id = 0;

FArkSourceControlCommand::FArkSourceControlCommand(const FSourceControlOperationRef& in_operation, FSourceControlChangelistPtr in_changelist, const FSourceControlOperationComplete& in_operation_complete_delegate, EConcurrency::Type in_concurrency, const TArray<FString>& in_files)
	:operation(in_operation)
	,changelist(in_changelist)
	,operation_complete_delegate(in_operation_complete_delegate)
	,async(in_concurrency == EConcurrency::Type::Asynchronous)
{

	FArkSourceControlProvider& provider = FArkSourceControlModule::Get().provider;

	relative_paths = provider.get_relative_paths(in_files);

	last_id += 1;
	id = last_id;
}

FArkSourceControlCommand::~FArkSourceControlCommand() {
}

bool FArkSourceControlCommand::execute()
{
	ECommandResult::Type result;

	bool is_finished = execute_internal(result);

	if (is_finished) {
		operation_complete_delegate.ExecuteIfBound(operation, result);
	}

	return is_finished;
}

void FArkSourceControlCommand::abort() {
	FMessageLog("SourceControl").Error(FText::FromString(FString::Printf(TEXT("%s aborted"), *operation.Get().GetName().ToString())));
	operation_complete_delegate.ExecuteIfBound(operation, ECommandResult::Type::Failed);
}

Ark_Plugin_Array_String FArkSourceControlCommand::get_relative_paths_as_array_string() {
	auto* plugin = get_plugin(); 

	Ark_Plugin_Array_String relative_path_array = ark_plugin_c_array_string_talloc(plugin, relative_paths.Num());
	for (int32 i = 0; i != relative_paths.Num();  ++i) {
		ark_plugin_c_array_string_set(&relative_path_array, i, ark_plugin_c_string_talloc(plugin, TCHAR_TO_UTF8(*relative_paths[i])));
	}
	return relative_path_array;
}

FArkSourceControlProvider& FArkSourceControlCommand::get_provider() {
	return FArkSourceControlModule::Get().provider;
}

Ark_Plugin* FArkSourceControlCommand::get_plugin() {
	return FArkSourceControlModule::Get().provider.plugin;
}

void FArkSourceControlCommand::push_request_id(uint64_t request_id) {
	FArkSourceControlModule::Get().provider.push_request(this, request_id);
}

bool FArkSourceControlCommandConnect::execute_internal(ECommandResult::Type& out_result)
{
	auto& provider = get_provider();

	if (provider.ark_proc_handle.IsValid() && FPlatformProcess::IsProcRunning(provider.ark_proc_handle)) {
		// Queue request to get ws cls
		provider.update_ws_cls();
		out_result = ECommandResult::Succeeded;
		return true;
	}
	
	if (!requested_ping) {
		time_abort_wait_for_ping = -1;
		requested_ping = true;
		if (ark_plugin_c_is_connected(provider.plugin)) {
			push_request_id(ark_plugin_c_request_ping(provider.plugin));
		}
		else {
			received_ping = true; // Mark it as if we had received it
		}
	}
	else {
		if (time_abort_wait_for_ping < 0) {
			if (GEditor && GEditor->IsInitialized()) {
				constexpr double MAX_TIME_WAIT_PING_RESPONSE = 1.0;
				time_abort_wait_for_ping = FPlatformTime::Seconds() + MAX_TIME_WAIT_PING_RESPONSE;
			}
		}
		else if (FPlatformTime::Seconds() >= time_abort_wait_for_ping) {
			received_ping = true; // Pretend we received it so that it gets launched bellow
		}
	}

	if (received_ping) {
		// If we received ping but ark_proc_handle is not valid, then we try to launch it
		out_result = provider.launch_ark() ? ECommandResult::Succeeded : ECommandResult::Failed;
		return true;
	}
		
	return false;
}

void FArkSourceControlCommandConnect::handle_response(Ark_Plugin_Response* response) {
	received_ping = true;

	if (response->ping.process_id > 0) {
		get_provider().connect_to_ark((uint32)response->ping.process_id);
	}
}

bool FArkSourceControlCommandUpdateStatus::execute_internal(ECommandResult::Type& out_result) {
	if (num_requests_sent == 0)	{
		
		FUpdateStatus* update_status = (FUpdateStatus*)operation.operator->();

		Ark_Plugin_Array_String relative_paths_array = get_relative_paths_as_array_string();

		push_request_id(ark_plugin_c_request_file_state(get_plugin(), relative_paths_array));

		num_requests_sent += 1;

		if (update_status->ShouldUpdateHistory()) {
			push_request_id(ark_plugin_c_request_file_history(get_plugin(), relative_paths_array));

			num_requests_sent += 1;
		}
	}

	bool is_finished = num_requests_sent > 0 && num_requests_sent == num_responses_received;

	if (is_finished) {
		out_result = ECommandResult::Succeeded;
	}

	return is_finished;
}

void FArkSourceControlCommandUpdateStatus::handle_response(Ark_Plugin_Response* response) {
	num_responses_received += 1;

	auto& provider = get_provider();

	if (response->type == Request_Type_FILE_STATE) {
		ensure(relative_paths.Num() == response->file_state.files.count);
		for (int32 i = 0; i != relative_paths.Num(); ++i) {
			auto* file_state = provider.get_file_state(relative_paths[i]);
			file_state->update(*(response->file_state.files.data + i));
		}
	} else if (response->type == Request_Type_FILE_HISTORY) {
		ensure(relative_paths.Num() == response->file_history.histories.count);
		for (int32 i = 0; i != relative_paths.Num(); ++i) {
			auto* file_state = provider.get_file_state(relative_paths[i]);
			file_state->update(*(response->file_history.histories.data + i));
		}
	}
}

bool FArkSourceControlCommandUpdateChangelistsStatus::execute_internal(ECommandResult::Type& out_result) {
	if (!has_sent_request) {
		has_sent_request = true;
		push_request_id(ark_plugin_c_request_ws_cl_list(get_plugin()));
	}

	if (has_received_response) {
		out_result = ECommandResult::Succeeded;
	}
	
	return has_received_response;
}

void FArkSourceControlCommandUpdateChangelistsStatus::handle_response(Ark_Plugin_Response* response) {
	has_received_response = true;

	auto& provider = get_provider();

	provider.clear_all_ws_cls();

	for (int32 ws_cl_index = 0; ws_cl_index != response->ws_cl_list.ws_cls.count; ++ws_cl_index) {

		Ark_Plugin_Ws_Cl* plugin_ws_cl = response->ws_cl_list.ws_cls.data + ws_cl_index;

		FSourceControlChangelistStateRef ws_cl_ref = provider.get_ws_cl_or_add(plugin_ws_cl->ws_cl_id);

		FArkSourceControlChangelistState& ws_cl = static_cast<FArkSourceControlChangelistState&>(ws_cl_ref.Get());

		ws_cl.update(*plugin_ws_cl);
	}
}

bool FArkSourceControlCommandCheckOut::execute_internal(ECommandResult::Type& out_result) {
	if (!has_sent_request) {
		has_sent_request = true;
		push_request_id(ark_plugin_c_request_file_lock (get_plugin(), get_relative_paths_as_array_string()));
	}

	if (has_received_response) {
		out_result = ECommandResult::Succeeded;
	}

	return has_received_response;
}

void FArkSourceControlCommandCheckOut::handle_response(Ark_Plugin_Response* response) {
	has_received_response = true;
		
	if (!ensure(relative_paths.Num() == response->file_lock.files.count)) {
		has_sent_request = false;
		has_received_response = false;
		return;
	}

	auto& provider = get_provider();

	for (int32 i = 0; i != relative_paths.Num(); ++i) {
		FArkSourceControlState* file_state = provider.get_file_state(relative_paths[i]);
		Ark_Plugin_File* file = response->file_lock.files.data + i;
		file_state->update(*file);
		if (file->asset_id != 0 && ((file->lock & File_Lock_LOCK_OWNER) == 0)) {
			provider.show_notification_request_lock_ownership(relative_paths[i], file_state->lock_owner);
		}
	}
}

bool FArkSourceControlCommandRevert::execute_internal(ECommandResult::Type& out_result) {
	if (!has_sent_request) {
		has_sent_request = true;

		push_request_id(ark_plugin_c_request_file_revert(get_plugin(), get_relative_paths_as_array_string()));
	}

	if (has_received_response) {
		out_result = ECommandResult::Succeeded;
	}

	return has_received_response;
}

void FArkSourceControlCommandRevert::handle_response(Ark_Plugin_Response* response) {
	has_received_response = true;
	
	if (!ensure(relative_paths.Num() == response->file_revert.files.count)) {
		has_sent_request = false;
		has_received_response = false;
		return;
	}

	auto& provider = get_provider();

	for (int32 i = 0; i != relative_paths.Num(); ++i) {
		auto* file_state = provider.get_file_state(relative_paths[i]);
		file_state->update(*(response->file_revert.files.data + i));
	}
}

bool FArkSourceControlCommandCheckIn::execute_internal(ECommandResult::Type& out_result) {
	if (!sent_request) {
		sent_request = true;

		auto* plugin = get_plugin();

		Ark_Plugin_Array_String relative_path_array = get_relative_paths_as_array_string();

		FCheckIn* check_in = (FCheckIn*)operation.operator->();
		Ark_Plugin_String comment = ark_plugin_c_string_talloc(plugin, TCHAR_TO_UTF8(*check_in->GetDescription().ToString()));

		FArkSourceControlChangelist* ws_cl = (FArkSourceControlChangelist*)changelist.Get();
		if (ws_cl != nullptr) {
			push_request_id(ark_plugin_c_request_ws_cl_commit(plugin, comment, ws_cl->ws_cl_id, check_in->GetKeepCheckedOut()));
		}
		else {
			push_request_id(ark_plugin_c_request_file_commit(plugin, comment, relative_path_array, check_in->GetKeepCheckedOut()));
		}
	}

	if (received_response) {
		out_result = success ? ECommandResult::Succeeded : ECommandResult::Failed;
		auto& provider = get_provider();
		provider.update_ws_cls();
		if (relative_paths.Num()) {
			provider.force_update_relative_paths(relative_paths);
		}
	}

	return received_response;
}

void FArkSourceControlCommandCheckIn::handle_response(Ark_Plugin_Response* response) {
	success = response->error == 0;
	received_response = true;
}


bool FArkSourceControlCommandNewChangelist::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		FNewChangelist* new_changelist = (FNewChangelist*)operation.operator->();
		auto* plugin = get_plugin();
		Ark_Plugin_String comment = ark_plugin_c_string_talloc(plugin, TCHAR_TO_UTF8(*new_changelist->GetDescription().ToString()));
		push_request_id(ark_plugin_c_request_ws_cl_create(plugin, comment, get_relative_paths_as_array_string()));
	}

	if (received_response) {
		out_result = ECommandResult::Type::Succeeded;
	}

	return received_response;
}

void FArkSourceControlCommandNewChangelist::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	if (response->ws_cl_create.ws_cl_id > 0) {
		get_provider().update_ws_cls();
	}
}

bool FArkSourceControlCommandDeleteChangelist::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		auto* plugin = get_plugin();
		
		FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get();

		push_request_id(ark_plugin_c_request_ws_cl_delete(plugin, ws_cl_ref->ws_cl_id));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
	}

	return received_response;
}

void FArkSourceControlCommandDeleteChangelist::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = response->error == 0;

	if (success) {
		FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get();
		get_provider().delete_ws_cl(ws_cl_ref->ws_cl_id);
	}
}

bool FArkSourceControlCommandEditChangelist::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		auto* plugin = get_plugin();
		
		FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get();
		FEditChangelist& edit_changelist = (FEditChangelist&)operation.Get();
		Ark_Plugin_String comment = ark_plugin_c_string_talloc(plugin, TCHAR_TO_UTF8(*edit_changelist.GetDescription().ToString()));
		push_request_id(ark_plugin_c_request_ws_cl_edit(plugin, ws_cl_ref->ws_cl_id, comment));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
	}

	return received_response;
}

void FArkSourceControlCommandEditChangelist::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = response->error == 0;

	if (success) {
		FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get();
		FArkSourceControlChangelistState& ws_cl_state = (FArkSourceControlChangelistState&)get_provider().get_ws_cl_or_add(ws_cl_ref->ws_cl_id).Get();
		FEditChangelist& edit_changelist = (FEditChangelist&)operation.Get();
		ws_cl_state.comment = edit_changelist.GetDescription().ToString();
	}
}

bool FArkSourceControlCommandMoveToChangelist::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		auto* plugin = get_plugin();
		
		FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get();
		push_request_id(ark_plugin_c_request_ws_cl_move_files(plugin, ws_cl_ref->ws_cl_id, get_relative_paths_as_array_string()));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
		get_provider().update_ws_cls();
	}

	return received_response;
}

void FArkSourceControlCommandMoveToChangelist::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = response->error == 0;
}

FArkSourceControlCommandSync::FArkSourceControlCommandSync(const FSourceControlOperationRef& in_operation, FSourceControlChangelistPtr in_changelist, const FSourceControlOperationComplete& in_operation_complete_delegate, EConcurrency::Type in_concurrency, const TArray<FString>& in_files)
:FArkSourceControlCommand(in_operation, in_changelist, in_operation_complete_delegate, in_concurrency, in_files)
{
	// There's a lot of things that are messy with this, or I'm missing something.
	// So when you do Tool -> Sync Content, it's going to us to sync a bunch of 
	// Content folders. This means that it wants to get the latest on all these folders.
	// However, if you select root folder and do Version Control -> Sync, it's going to go through 
	// all the files and try to get their info so that it then asks all the files to update.
	// There's supposed to be a get head revision flag, but I don't see anybody setting it on...
	// So here's what we'll do: If while going through these folders, if we see Content/, so ignore all the
	// paths and do sync to latest.. We even set the head revision flag because why not?!
	for (const FString& relative_path : relative_paths) {
		if (relative_path == TEXT("Content/")) {
			FSync& sync = (FSync&)operation.Get();
			sync.SetHeadRevisionFlag(true);
			break;
		}
	}
}

bool FArkSourceControlCommandSync::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		auto& provider = get_provider();
		
		FSync& sync = (FSync&)operation.Get();
		
		uint32_t highest_cl_id = 0; // 0 means latest

		if (!sync.IsHeadRevisionFlagSet()) {
			if (FArkSourceControlChangelist* ws_cl_ref = (FArkSourceControlChangelist*)changelist.Get()) {
				highest_cl_id = ws_cl_ref->cl_id;
			}
			
			for (int32 i = 0; i != relative_paths.Num(); ++i) {
				if (FArkSourceControlState* file_state = provider.get_file_state(relative_paths[i])) {
					if (highest_cl_id < file_state->file.head_cl_id) {
						highest_cl_id = file_state->file.head_cl_id;
					}
				}
			}
		}
		push_request_id(ark_plugin_c_request_get_cl(provider.plugin, highest_cl_id));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
		get_provider().update_ws_cls();
	}

	return received_response;
}

void FArkSourceControlCommandSync::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = response->error == 0;
}


bool FArkSourceControlCommandGetFile::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;
		auto& provider = get_provider();
		
		FGetFile& get_file = (FGetFile&)operation.Get();

		uint32_t cl_id = (uint32_t)FPlatformString::Atoi64(*get_file.GetChangelistNumber());
		uint32_t cl_revision = (uint32_t)FPlatformString::Atoi64(*get_file.GetRevisionNumber());
		
		// Turns out that this command only asks for 1 asset at a time, so we can probably revisit this later and remove the array
		Ark_Plugin_Array_File_Get file_get_array;
		file_get_array.count = 1;
		file_get_array.data = (Ark_Plugin_File_Get*)FMemory::MallocZeroed(file_get_array.count * sizeof(Ark_Plugin_File_Get));
		
		FArkSourceControlState* file_state = provider.get_file_state(get_file.GetDepotFilePath());
		Ark_Plugin_File_Get* file_get = file_get_array.data;
		file_get->cl_id = cl_id;
		file_get->cl_revision = cl_revision;
		file_get->asset_id = file_state->file.asset_id;

		push_request_id(ark_plugin_c_request_file_get(provider.plugin, file_get_array));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
	}

	return received_response;
}

void FArkSourceControlCommandGetFile::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = true;
}

bool FArkSourceControlCommandResolve::execute_internal(ECommandResult::Type& out_result) {

	if (!sent_request) {
		sent_request = true;

		push_request_id(ark_plugin_c_request_file_resolve(get_plugin(), get_relative_paths_as_array_string()));
	}

	if (received_response) {
		out_result = success ? ECommandResult::Type::Succeeded : ECommandResult::Type::Failed;
	}

	return received_response;
}

void FArkSourceControlCommandResolve::handle_response(Ark_Plugin_Response* response) {
	received_response = true;

	success = true;
}


