﻿// Copyright Nuno Afonso

#include "ArkSourceControlSettings.h"

#include "Misc/ConfigCacheIni.h"
#include "SourceControlHelpers.h"
#include "Ark_Plugin.h"

namespace ArkSettingsConstants
{
	static const FString SettingsSection = TEXT("ArkSourceControl.ArkSourceControlSettings");
}

#define SETTINGS_VAR(A) TEXT(#A)

void FArkSourceControlSettings::load_settings() {
	FScopeLock scope_lock(&critical_section);
	const FString& ini_file = SourceControlHelpers::GetSettingsIni();
	GConfig->GetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(binary_path), binary_path, ini_file);
	if (binary_path == "") {
		binary_path = try_to_find_binary_path();
	}
	GConfig->GetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(workspace_path), workspace_path, ini_file);
	if (workspace_path == "") {
		workspace_path = choose_best_workspace_path();
	}
	GConfig->GetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(email), email, ini_file);
	GConfig->GetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(host), host, ini_file);
	GConfig->GetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(use_built_in_server), use_built_in_server, ini_file);
	GConfig->GetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(add_default_ignore), add_default_ignore, ini_file);
	GConfig->GetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(add_default_locks), add_default_locks, ini_file);
}

void FArkSourceControlSettings::save_settings() const {
	FScopeLock scope_lock(&critical_section);
	const FString& ini_file = SourceControlHelpers::GetSettingsIni();
	GConfig->SetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(binary_path), *binary_path, ini_file);
	GConfig->SetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(workspace_path), *workspace_path, ini_file);
	GConfig->SetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(email), *email, ini_file);
	GConfig->SetString(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(host), *host, ini_file);
	GConfig->SetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(use_built_in_server), use_built_in_server, ini_file);
	GConfig->SetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(add_default_ignore), add_default_ignore, ini_file);
	GConfig->SetBool(*ArkSettingsConstants::SettingsSection, SETTINGS_VAR(add_default_locks), add_default_locks, ini_file);
}

FString FArkSourceControlSettings::try_to_find_binary_path() {
#if PLATFORM_WINDOWS
	FString file_paths_to_try[] =
		{ TEXT("C:\\Ark-Vcs\\ark.exe")
		, TEXT("C:\\Program Files\\Ark-Vcs\\ark.exe")
		, TEXT("C:\\Program Files (x86)\\Ark-Vcs\\ark.exe")
		};
#elif PLATFORM_MAC
	FString file_paths_to_try[] =
			{ TEXT("/Applications/Ark Vcs.app/Contents/MacOS/ark-macos")
			, TEXT("/usr/bin/ark-macos")
			, TEXT("/usr/local/bin/ark-macos")
			};
#elif PLATFORM_LINUX
	FString file_paths_to_try[] =
		{ TEXT("/usr/bin/ark-macos")
		, TEXT("/usr/local/bin/ark-macos")
		};
#endif
	
	for (FString& file_path : file_paths_to_try) {
		if (IFileManager::Get().FileExists(*file_path)) {
			return file_path;
		}
	}

	return "";
}

FString FArkSourceControlSettings::choose_best_workspace_path() {
	// Prioritize game folder 
	FString game_path = FPaths::ConvertRelativePathToFull(FPaths::ProjectDir());
	FString path = FString::Printf(TEXT("%s%s"), *game_path, ANSI_TO_TCHAR(ARK_FOLDER));

	if (IFileManager::Get().DirectoryExists(*path)) {
		return game_path;
	}

	// If not, try engine path
	FString engine_path = FPaths::ConvertRelativePathToFull(FPaths::EngineDir()); 
	path = FString::Printf(TEXT("%s%s"), *engine_path, ANSI_TO_TCHAR(ARK_FOLDER));
	if (IFileManager::Get().DirectoryExists(*path)) {
		return engine_path;
	}

	// Then try to see if the game path is tied to the engine
	if (engine_path.EndsWith("Engine/"))
	{
		FString engine_root_path = engine_path.LeftChop(FString("Engine/").Len());

		if (game_path.StartsWith(engine_root_path)) {
			return engine_root_path;
		}
	}

	// Otherwise default to game folder, user can always choose another one
	return game_path;
}

FString FArkSourceControlSettings::get_port() {
	if (!host.Len()) {
		return host;
	}
	int32 bracket_index = -1;
	int32 colon_index = -1;
	if (host.FindLastChar(']', bracket_index)) {
		// It should be ipv6
		if (host.FindLastChar(':', colon_index)) {
			if (colon_index > bracket_index) {
				return host.RightChop(colon_index+1);
			}
		}
	}
	else {
		// Should be ipv4
		if (host.FindLastChar(':', colon_index)) {
			return host.RightChop(colon_index+1);
		}
	}
	return FString();
}

void FArkSourceControlSettings::set_localhost_with_port(const FString& port) {
	host = port.Len() ? FString::Printf(TEXT("localhost:%s"), *port) : TEXT("localhost");
}



#undef SETTINGS_VAR
